#ifndef _epl_public_h
#define _epl_public_h

#include <stdint.h>
#include "epl_cn_xpec_regdef.h"

/* pragma unpack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_PACK_1(EPL_PUBLIC)
#endif

/* EPL Message Type Ids */
#define EPLCN_MSG_TYPE_SOC                (0x01)    /* Start of Cycle */
#define EPLCN_MSG_TYPE_PREQ               (0x03)    /* Poll Request */
#define EPLCN_MSG_TYPE_PRES               (0x04)    /* Poll Response */
#define EPLCN_MSG_TYPE_SOA                (0x05)    /* Start of Asynchronous */
#define EPLCN_MSG_TYPE_ASND               (0x06)    /* Asynchronous Send */

/* MN and CN states */
#define EPLCN_NMT_GS_OFF                  (0x00)    /* 0000 0000b */
#define EPLCN_NMT_GS_POWERED              (0x08)    /* xxxx 1xxxb, super state */
#define EPLCN_NMT_GS_INITIALISATION       (0x09)    /* xxxx 1001b, super state */
#define EPLCN_NMT_GS_INITIALISING         (0x19)    /* 0001 1001b */
#define EPLCN_NMT_GS_RESET_APPLICATION    (0x29)    /* 0010 1001b */
#define EPLCN_NMT_GS_RESET_COMMUNICATION  (0x39)    /* 0011 1001b */
#define EPLCN_NMT_GS_RESET_CONFIGURATION  (0x79)    /* 0111 1001b */
#define EPLCN_NMT_GS_COMMUNICATING        (0x0c)    /* xxxx 11xxb, super state */

/* CN states */
#define EPLCN_NMT_CS_NOT_ACTIVE           (0x1c)    /* 0001 1100b */
#define EPLCN_NMT_CS_EPL_MODE             (0x0d)    /* xxxx 1101b, super state */
#define EPLCN_NMT_CS_PRE_OPERATIONAL_1    (0x1d)    /* 0001 1101b */
#define EPLCN_NMT_CS_PRE_OPERATIONAL_2    (0x5d)    /* 0101 1101b */
#define EPLCN_NMT_CS_READY_TO_OPERATE     (0x6d)    /* 0110 1101b */
#define EPLCN_NMT_CS_OPERATIONAL          (0xfd)    /* 1111 1101b */
#define EPLCN_NMT_CS_STOPPED              (0x4d)    /* 0100 1101b */
#define EPLCN_NMT_CS_BASIC_ETHERNET       (0x1e)    /* 0001 1110b */

/* MN states */
#define EPLCN_NMT_MS_NOT_ACTIVE           (0x1c)    /* 0001 1100b, same as CN satte code */
#define EPLCN_NMT_MS_EPL_MODE             (0x0d)    /* xxxx 1101b */
#define EPLCN_NMT_MS_PRE_OPERATIONAL_1    (0x1d)    /* 0001 1101b, same as CN state code  */
#define EPLCN_NMT_MS_PRE_OPERATIONAL_2    (0x5d)    /* 0101 1101b, same as CN state code */
#define EPLCN_NMT_MS_READY_TO_OPERATE     (0x6d)    /* 0110 1101b, same as CN state code */
#define EPLCN_NMT_MS_OPERATIONAL          (0xfd)    /* 1111 1101b, same as CN state code */

#define EPLCN_NMT_OPERATIONAL_FLAG_MASK   (0x80)


/* SoA requested services */
#define EPLCN_SOA_SVC_NO_SERVICE                        (0x00)
#define EPLCN_SOA_SVC_IDENT_REQUEST                     (0x01)
#define EPLCN_SOA_SVC_STATUS_REQUEST                    (0x02)
#define EPLCN_SOA_SVC_NMT_REQUEST_INVITE                (0x03)
#define EPLCN_SOA_SVC_UNSPECIFIED_INVITE                (0xFF)

#define EPLCN_SOA_SVC_MANUF_START                       (0xa0)
#define EPLCN_SOA_SVC_MANUF_END                         (0xfe)

/* requestable ASnd ServiceIds */
#define EPLCN_ASND_SVC_IDENT_RESPONSE                   (0x01)
#define EPLCN_ASND_SVC_STATUS_RESPONSE                  (0x02)
#define EPLCN_ASND_SVC_NMT_REQUEST                      (0x03)
#define EPLCN_ASND_SVC_NMT_COMMAND                      (0x04)
#define EPLCN_ASND_SVC_SDO                              (0x05)

#define EPLCN_ASND_SVC_MANUF_START                      (0xa0)
#define EPLCN_ASND_SVC_MANUF_END                        (0xfe)

/* Plain state commands */
#define EPLCN_ASND_CMD_NMT_START_NODE                   (0x21)
#define EPLCN_ASND_CMD_NMT_STOP_NODE                    (0x22)
#define EPLCN_ASND_CMD_NMT_ENTER_PRE_OP2                (0x23)
#define EPLCN_ASND_CMD_NMT_ENABLE_RDY_TO_OP             (0x24)
#define EPLCN_ASND_CMD_NMT_RESET_NODE                   (0x28)
#define EPLCN_ASND_CMD_NMT_RESET_COMMUNICATION          (0x29)
#define EPLCN_ASND_CMD_NMT_RESET_CONFIGURATION          (0x2A)
#define EPLCN_ASND_CMD_NMT_SW_RESET                     (0x2B)

/* Extended NMT state commands */
#define EPLCN_ASND_CMD_NMT_START_NODE_EX                (0x41)
#define EPLCN_ASND_CMD_NMT_STOP_NODE_EX                 (0x42)
#define EPLCN_ASND_CMD_NMT_ENTER_PRE_OP2_EX             (0x43)
#define EPLCN_ASND_CMD_NMT_ENABLE_RDY_TO_OP_EX          (0x44)
#define EPLCN_ASND_CMD_NMT_RESET_NODE_EX                (0x48)
#define EPLCN_ASND_CMD_NMT_RESET_COMMUNICATION_EX       (0x49)
#define EPLCN_ASND_CMD_NMT_RESET_CONFIGURATION_EX       (0x4A)
/* NMT Managing Node Commands */
#define EPLCN_ASND_CMD_NET_HOST_NAME_SET                (0x62)
#define EPLCN_ASND_CMD_FLUSH_ARP_ENTRY                  (0x63)
/* NMT info services */
#define EPLCN_ASND_CMD_PUBLISH_CONFIGURED_CN            (0x80)
#define EPLCN_ASND_CMD_PUBLISH_ACTIVE_CN                (0x90)
#define EPLCN_ASND_CMD_PUBLISH_PRE_OP1                  (0x91)
#define EPLCN_ASND_CMD_PUBLISH_PRE_OP2                  (0x92)
#define EPLCN_ASND_CMD_PUBLISH_RDY_TO_OP                (0x93)
#define EPLCN_ASND_CMD_PUBLISH_OPERATIONAL              (0x94)
#define EPLCN_ASND_CMD_PUBLISH_STOPPED                  (0x95)
#define EPLCN_ASND_CMD_PUBLISH_EMERGENCY_NEW            (0xA0)
#define EPLCN_ASND_CMD_PUBLISH_TIME                     (0xB0)

#define EPLCN_ASND_CMD_INVALID_SERVICE                  (0xFF)


/* General Purpose Constants */

#define EPLCN_C_ADR_BROADCAST                           (0xFF)      /* EPL broadcast address */
#define EPLCN_C_ADR_INVALID                             (0x00)      /* invalid EPL address */
#define EPLCN_C_ADR_MN_DEF_NODE_ID                      (0xF0)      /* EPL default address of MN */
#define EPLCN_C_ADR_DUMMY_NODE_ID                       (0xFC)      /* EPL dummy node address */
#define EPLCN_C_ADR_DIAG_DEF_NODE_ID                    (0xFD)      /* EPL default address of diagnostic device */
#define EPLCN_C_ADR_RT1_DEF_NODE_ID                     (0xFE)      /* EPL default address of router 1 */
#define EPLCN_C_DLL_ASND_PRIO_NMTRQST                   (7)         /* increased ASnd request priority to be used by NMT requests */
#define EPLCN_C_DLL_ASND_PRIO_STD                       (0)         /* standard ASnd request priority */
#define EPLCN_C_DLL_ETHERTYPE_EPL                       (0x88AB)
#define EPLCN_C_DLL_ETHERTYPE_IP                        (0x0800)
#define EPLCN_C_DLL_ISOCHR_MAX_PAYL                     (1490)      /* maximum size of of PReq and PRes payload */
#define EPLCN_C_DLL_MAX_ASYNC_MTU                       (1500)      /* maximum asynchronous payload in bytes */
#define EPLCN_C_DLL_MAX_PAYL_OFFSET                     (1499)      /* maximum offset of Ethernet frame payload */
#define EPLCN_C_DLL_MAX_RS                              (7)
#define EPLCN_C_DLL_MIN_ASYNC_MTU                       (282)       /* minimum asynchronous payload bytes */
#define EPLCN_C_DLL_MIN_PAYL_OFFSET                     (45)        /* minimum offset of Ethernet frame payload */
#define EPLCN_C_DLL_MULTICAST_ASND                      {0x01,0x11,0x1E,0x00,0x00,0x04} /* EPL ASnd multicast MAC address */
#define EPLCN_C_DLL_MULTICAST_PRES                      {0x01,0x11,0x1E,0x00,0x00,0x02} /* EPL PRes multicast MAC address */
#define EPLCN_C_DLL_MULTICAST_SOA                       {0x01,0x11,0x1E,0x00,0x00,0x03} /* EPL SoA multicast MAC address */
#define EPLCN_C_DLL_MULTICAST_SOC                       {0x01,0x11,0x1E,0x00,0x00,0x01} /* EPL SoC multicast MAC address */
#define EPLCN_C_IP_ADR_INVALID                          {0x00,0x00,0x00,0x00}
#define EPLCN_C_IP_INVALID_MTU                          (0)         /* invalid MTU size used to indicate no change */
#define EPLCN_C_IP_MAX_MTU                              (1518)      /* maximum size in bytes of the IP stack which must be processed */
#define EPLCN_C_IP_MIN_MTU                              (300/*ns*/) /* minimum size in bytes of the IP stack which must be processed */
#define EPLCN_C_DLL_T_IFG                               (120/*ns*/) /* Ethernet Interframe Gap (120ns) */
#define EPLCN_C_DLL_T_MIN_FRAME                         (640/*ns*/) /* size of minimum Ethernet frame (without preamble) (640ns) */
#define EPLCN_C_DLL_T_PREAMBLE                          (80/*ns*/)  /* Size of Ethernet frame preamble (80ns) */
#define EPLCN_C_SDO_EPL_PORT                            (3819)      /* port to be used EPL specific UDP/IP frames */
#define EPLCN_C_DLL_PREOP1_START_CYCLES                 (10)        /* number of unassigned SoA frames at start of NMT_MS_PRE_OPERATIONAL_1 */
#define EPLCN_C_NMT_STATREQ_CYCLE                       (5)         /* StatusRequest cycle time to be applied to AsyncOnly CNs */

typedef uint32_t EPLCN_RESULT;
typedef uint8_t  EPLCN_NMT_STATE;

typedef __PACKED_PRE struct EPLCN_POLL_REQ_Ttag
{
  uint8_t   bMsgType;
  uint8_t   bDestination;
  uint8_t   bSource;
  uint8_t   bReserved;
  uint8_t   bFlags;
  uint8_t   bPrRs;
  uint8_t   bPDOVersion;
  uint8_t   bReserved2;
  uint16_t  usSize;
  uint8_t   abData[EPLCN_C_DLL_ISOCHR_MAX_PAYL];
} __PACKED_POST EPLCN_POLL_REQ_T;

typedef __PACKED_PRE struct EPLCN_POLL_REQ_HEADER_Ttag
{
  uint8_t   bMsgType;
  uint8_t   bDestination;
  uint8_t   bSource;
  uint8_t   bReserved;
  uint8_t   bFlags;
  uint8_t   bPrRs;
} __PACKED_POST EPLCN_POLL_REQ_HEADER_T;

typedef __PACKED_PRE struct EPLCN_POLL_RESP_Ttag
{
  uint8_t   bMsgType;
  uint8_t   bDestination;
  uint8_t   bSource;
  uint8_t   bNMTStatus;
  uint8_t   bFlags;
  uint8_t   bPrRs;
  uint8_t   bPDOVersion;
  uint8_t   bReserved2;
  uint16_t  usSize;
  uint8_t   abData[EPLCN_C_DLL_ISOCHR_MAX_PAYL];
} __PACKED_POST EPLCN_POLL_RESP_T;

typedef __PACKED_PRE struct EPLCN_POLL_RESP_HEADER_Ttag
{
  uint8_t   bMsgType;
  uint8_t   bDestination;
  uint8_t   bSource;
  uint8_t   bNMTStatus;
  uint8_t   bFlags;
} __PACKED_POST EPLCN_POLL_RESP_HEADER_T;

#define EPLCN_FRAME_PREQ_FLAG_RD           0x01
#define EPLCN_FRAME_PREQ_FLAG_EA           0x04
#define EPLCN_FRAME_PRES_FLAG_RD           0x01
#define EPLCN_FRAME_PRES_FLAG_EN           0x10


#define EPLCN_PRES_FLAG_RS_MASK     (0x07)  /* Mask for Request to Send */
#define EPLCN_PRES_FLAG_PR_MASK     (0x38)  /* Mask for Priority */
#define EPLCN_PRES_RS_SHIFT         (0)
#define EPLCN_PRES_PR_SHIFT         (3)

#define EPLCN_RS_PR7(cnt) ((0<<EPLCN_PRES_PR_SHIFT)|((cnt>7?7:cnt)<<EPLCN_PRES_RS_SHIFT))
#define EPLCN_RS_PR0(cnt) ((0<<EPLCN_PRES_PR_SHIFT)|((cnt>7?7:cnt)<<EPLCN_PRES_RS_SHIFT))

typedef uint8_t EPLCN_MAC_ADDR_T[6];

typedef __PACKED_PRE struct EPLCN_ETH_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
} __PACKED_POST EPLCN_ETH_FRAME_T;

typedef __PACKED_PRE struct EPLCN_POLL_RESP_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
  EPLCN_POLL_RESP_T       tPollResp;
} __PACKED_POST EPLCN_POLL_RESP_FRAME_T;

typedef __PACKED_PRE struct EPLCN_POLL_REQ_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
  EPLCN_POLL_REQ_T        tPollReq;
} __PACKED_POST EPLCN_POLL_REQ_FRAME_T;

typedef __PACKED_PRE struct EPLCN_HEADER_Ttag
{
  uint8_t bMsgType;
  uint8_t bDestination;
  uint8_t bSource;
  uint8_t bNMTStatus; /* not on SoC */
} __PACKED_POST EPLCN_HEADER_T;

typedef __PACKED_PRE struct EPLCN_SOA_Ttag
{
  uint8_t             bMsgType;
  uint8_t             bDestination;
  uint8_t             bSource;
  uint8_t             bNMTStatus;
  /* flag set */
  uint8_t             bFlags;
  /* reserved */
  uint8_t             bReserved;
  /* requested service id */
  uint8_t             bRequestedServiceId;
  /* requested service target */
  uint8_t             bRequestedServiceTarget;
  /* EPL version */
  uint8_t             bEPLVersion;
} __PACKED_POST EPLCN_SOA_T;

#define EPLCN_FRAME_SOA_FLAG_ER           0x02
#define EPLCN_FRAME_SOA_FLAG_EA           0x04

typedef __PACKED_PRE struct EPLCN_ASND_Ttag
{
  /* message type */
  uint8_t             bMessageType;
  /* Destination EPL address */
  uint8_t             bDestination;
  /* Source EPL address */
  uint8_t             bSource;
  /* reserved */
  uint8_t             bServiceId;
} __PACKED_POST EPLCN_ASND_T;


typedef __PACKED_PRE struct EPLCN_ERROR_ENTRY_Ttag
{
  uint16_t usEntryType;
  uint16_t usErrorCode;
  uint32_t aulTimeStamp[2];
  uint32_t aulAddInformation[2];
} __PACKED_POST EPLCN_ERROR_ENTRY_T;

//typedef uint8_t EPLCN_ERROR_ENTRY_T[20];

typedef __PACKED_PRE struct EPLCN_IDENT_RESPONSE_Ttag
{
  /* reserved */
  uint8_t               bReserved0;
  /* frames to be sent */
  uint8_t               bPrRs;
  /* NMT status */
  uint8_t               bNmtStatus;
  /* reserved */
  uint8_t               bReserved;
  /* EPL ProfileVersion */
  uint8_t               bEplProfileVersion;
  /* reserved */
  uint8_t               bReserved2;
  /* Feature Flags */
  uint32_t              ulFeatureFlags;
  /* MTU */
  uint16_t              usMTU;
  /* PollInSize */
  uint16_t              usPollInSize;
  /* PollOutSize */
  uint16_t              usPollOutSize;
  /* Response Time */
  uint32_t              ulResponseTime;
  /* reserved */
  uint16_t              usReserved3;
  /* Device Type */
  uint32_t              ulDeviceType;
  /* Vendor Id */
  uint32_t              ulVendorId;
  /* Product Code */
  uint32_t              ulProductCode;
  /* Revision Number */
  uint32_t              ulRevisionNumber;
  /* Serial Number */
  uint32_t              ulSerialNumber;
  /* Vendor Specific Extension 1 */
  uint32_t              aulVendorExtension[2];
  /* Verify Configuration Date */
  uint32_t              ulVerifyConfigurationDate;
  /* Verify Configuration Time */
  uint32_t              ulVerifyConfigurationTime;
  /* Application Sw Date */
  uint32_t              ulApplicationSwDate;
  /* Application Sw Time */
  uint32_t              ulApplicationSwTime;
  /* IP Address */
  uint32_t              ulIpAddress;
  /* Subnet mask */
  uint32_t              ulSubnetMask;
  /* Default Gateway */
  uint32_t              ulDefaultGateway;
  /* Hostname */
  uint8_t               szHostName[32];
  /* Vendor Specific Extension 2 */
  uint8_t               abVendorExtension2[48];
} __PACKED_POST EPLCN_IDENT_RESPONSE_T;

#define EPLCN_FEATURE_FLAGS_ISOCHRONOUS                   0x00000001
#define EPLCN_FEATURE_FLAGS_SDO_VIA_UDP                   0x00000002
#define EPLCN_FEATURE_FLAGS_SDO_VIA_ASND                  0x00000004
#define EPLCN_FEATURE_FLAGS_SDO_VIA_PDO                   0x00000008
#define EPLCN_FEATURE_FLAGS_NMT_INFO_SERVICES             0x00000010
#define EPLCN_FEATURE_FLAGS_EXTENDED_NMT_STATE_COMMANDS   0x00000020
#define EPLCN_FEATURE_FLAGS_DYNAMIC_PDO_MAPPING           0x00000040
#define EPLCN_FEATURE_FLAGS_NMT_SERVICE_VIA_UDP           0x00000080
#define EPLCN_FEATURE_FLAGS_CONFIGURATION_MANAGER         0x00000100
#define EPLCN_FEATURE_FLAGS_MULTIPLEXED_ACCESS            0x00000200
#define EPLCN_FEATURE_FLAGS_NODEID_SETUP_BY_SOFTWARE      0x00000400
#define EPLCN_FEATURE_FLAGS_ROUTING_TYPE_1                0x00000800
#define EPLCN_FEATURE_FLAGS_ROUTING_TYPE_2                0x00001000

typedef uint8_t EPLCN_STATUS_RESPONSE_EVENT_T[20];

#define EPLCN_NUM_ENTRIES 73

typedef __PACKED_PRE struct EPLCN_STATUS_RESPONSE_Ttag
{
  /* flag set */
  uint8_t                           bFlags;
  /* Priority /Send */
  uint8_t                           bPrRs;
  /* NMT status */
  uint8_t                           bNmtStatus;
  /* reserved */
  uint8_t                           abReserved[3];
  /* static error bit field */
  uint8_t                           abStatusField[8];
  /* error events */
  EPLCN_STATUS_RESPONSE_EVENT_T     atEvents[EPLCN_NUM_ENTRIES];
} __PACKED_POST EPLCN_STATUS_RESPONSE_T;

#define EPLCN_STATUS_RESPONSE_FLAGS_EC      0x08
#define EPLCN_STATUS_RESPONSE_FLAGS_EN      0x10

typedef __PACKED_PRE struct EPLCN_IDENT_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
  EPLCN_ASND_T            tASnd;
  EPLCN_IDENT_RESPONSE_T  tIdentResponse;
} __PACKED_POST EPLCN_IDENT_FRAME_T;

typedef __PACKED_PRE struct EPLCN_ASND_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
  EPLCN_ASND_T            tASnd;
} __PACKED_POST EPLCN_ASND_FRAME_T;

typedef __PACKED_PRE struct EPLCN_SOA_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
  EPLCN_SOA_T             tSoA;
} __PACKED_POST EPLCN_SOA_FRAME_T;

typedef __PACKED_PRE struct EPLCN_EPLCN_STATUS_FRAME_Ttag
{
  /* ethernet header */
  EPLCN_MAC_ADDR_T        tDstAddr;         /* Destination MAC address */
  EPLCN_MAC_ADDR_T        tSrcAddr;         /* Source MAC address */
  uint16_t                usType;           /* Ethernet II type */
  EPLCN_ASND_T            tASnd;
  EPLCN_STATUS_RESPONSE_T tStatusResponse;
} __PACKED_POST EPLCN_STATUS_FRAME_T;

typedef __PACKED_PRE struct EPLCN_SoC_TIME_Ttag
{
  uint32_t              ulSeconds;
  uint8_t               bSign1;
  uint32_t              ulNanoseconds;
} __PACKED_POST EPLCN_SoC_TIME_T;

typedef __PACKED_PRE struct EPLCN_SOC_HEADER_Ttag
{
  EPLCN_HEADER_T        tFrameHead;
  /* flag set */
  uint8_t               bFlags;
  /* reserved */
  uint8_t               bReserved3;
  /* NetTime */
  EPLCN_SoC_TIME_T      tNetTime;
} __PACKED_POST EPLCN_SOC_HEADER_T;


typedef __PACKED_PRE struct EPLCN_NMT_COMMAND_Ttag
{
  /* command id */
  uint8_t               bNMTCommandId;
  /* reserved */
  uint8_t               bReserved;
  /* NMT command data */
  uint8_t               abData[EPLCN_C_DLL_MAX_PAYL_OFFSET-4];
} __PACKED_POST EPLCN_NMT_COMMAND_T;

typedef __PACKED_PRE struct EPLCN_NMT_REQUEST_Ttag
{
  /* command id */
  uint8_t               bNMTRequestedCommandId;
  /* requested command target */
  uint8_t               bNMTRequestedCommandTarget;
  /* requested command data */
  uint8_t               abData[EPLCN_C_DLL_MAX_PAYL_OFFSET-4];
} __PACKED_POST EPLCN_NMT_REQUEST_T;


typedef __PACKED_PRE union EPLCN_HEADER_FLAGS_Ttag
{
  EPLCN_HEADER_T            tHead;
  __PACKED_PRE struct {
    EPLCN_ASND_T            tHead;
    /* flag set */
    uint16_t                usFlags;
    /* NMT status */
    uint8_t                 bNmtStatus;
  } __PACKED_POST tASnd;
  EPLCN_SOA_T               tSoA;
  EPLCN_POLL_REQ_HEADER_T   tPReq;
  EPLCN_POLL_RESP_HEADER_T  tPRes;
} __PACKED_POST EPLCN_HEADER_FLAGS_T;

#define EPLCN_HEADER_FLAGS_RD 0x0001
#define EPLCN_HEADER_FLAGS_ER 0x0002
#define EPLCN_HEADER_FLAGS_EA 0x0004
#define EPLCN_HEADER_FLAGS_EN 0x0010
#define EPLCN_HEADER_FLAGS_MS 0x0020
#define EPLCN_HEADER_FLAGS_PS 0x0040
#define EPLCN_HEADER_FLAGS_MC 0x0080

/* pragma unpack */
#ifdef PRAGMA_UNPACK_ENABLE
#pragma PRAGMA_UNPACK_1(EPL_PUBLIC)
#endif


typedef struct EPLCN_INTERFACE_Ttag EPLCN_INTERFACE_T;
typedef EPLCN_INTERFACE_T* EPLCN_HANDLE_T;

typedef uint32_t EPLCN_RESULT_T;

enum EPLCN_TRANSPORT_TYPE_Ttag
{
  EPLCN_TRANSPORT_ETH,
  EPLCN_TRANSPORT_UDP
};

typedef uint32_t EPLCN_TRANSPORT_TYPE_T;

/**************************************************************************************************
 * Error Codes
 */
#define EPLCN_SUCCESS                           0x00000000
#define EPLCN_OUT_OF_MEMORY                     0x00030001
#define EPLCN_XC0_LOAD_ERROR                    0x00030002
#define EPLCN_XC1_LOAD_ERROR                    0x00030003
#define EPLCN_MEMORY_MAPPING_FAILED             0x00030004
#define EPLCN_COM0_IRQ_FAILED                   0x00030005
#define EPLCN_MSYNC0_IRQ_FAILED                 0x00030006
#define EPLCN_OSAL_INIT_FAILED                  0x00030007

#define EPLCN_FRAME_SIZE_INVALID                0x00031000
#define EPLCN_BUFFER_EXCHANGE_FAILED            0x00031001

#define EPLCN_RESULT_RESET_PENDING              0x00032000

#define EPLCN_TOO_MANY_FRAMES                   0x00033000

/**************************************************************************************************
 * Initialization
 */

typedef struct EPLCN_OSAL_Ttag
{
  unsigned long (*  osal_init)();
  void (*           osal_deinit)(unsigned long ulOsalHandle);

  void* (*          malloc)(unsigned long ulSize);
  void(*            free)(void*);

  void* (*          region_request)(unsigned long ulPhysAddr, unsigned long ulSize);
  void (*           region_release)(void* pvPhysMapping, unsigned long ulSize);

  unsigned long (*  irq_request)(unsigned long ulOsalHandle, unsigned long ulIrqSource, void(* pfnIrq)(unsigned long ulHandle, void* pvParam), void* pvParam);
  void (*           irq_release)(unsigned long ulIrqHandle);

  void (*           irq_enable)(unsigned long ulIrqHandle);
  void (*           irq_disable)(unsigned long ulIrqHandle);

  unsigned long (*  irq_lock)();
  void (*           irq_unlock)(unsigned long);
} EPLCN_OSAL_T;

#define MSK_EPL_SOC_TRIGGER_CONFIG_OUTPUT_ENABLE      0x01
#define MSK_EPL_SOC_TRIGGER_CONFIG_POLARITY           0x02

EPLCN_RESULT_T
EplCn_Initialize(
                  EPLCN_OSAL_T*   ptOsal,
                  uint8_t         bSoCTriggerConfig,
                  uint32_t        ulSoCTriggerDelay,
                  uint32_t        ulSoCTriggerLength,
                  int             fEnableColLed,
                  EPLCN_HANDLE_T* phEplHandle);

EPLCN_RESULT_T
EplCn_DeInitialize( EPLCN_HANDLE_T hEplHandle);


void
EplCn_SetMacAddress(
                    EPLCN_HANDLE_T    hEplHandle,
                    EPLCN_MAC_ADDR_T  tMacAddr);

void
EplCn_GetMacAddress(
                    EPLCN_HANDLE_T    hEplHandle,
                    uint8_t*          pbMacAddr);

EPLCN_RESULT_T
EplCn_Reset(        EPLCN_HANDLE_T    hEplHandle);


/**************************************************************************************************
 * EPL Cn Status
 */


EPLCN_STATUS_AREA_BASE_T*
EplCn_GetCounters(       EPLCN_HANDLE_T      hEplHandle);

#define EPLCN_MODE_LEGACY_ETHERNET    0
#define EPLCN_MODE_EPL                1

int EplCn_GetMode(  EPLCN_HANDLE_T    hEplHandle);

/** Link Status **/

typedef struct EPLCN_LINKSTATUS_INFO_Ttag
{
  struct
  {
    int fLinkIsUp;
    int uLinkSpeed;
    int fDuplex;
  } atPort[2];
} EPLCN_LINKSTATUS_INFO_T;

void EplCn_GetLinkStatus(EPLCN_HANDLE_T hEplHandle, EPLCN_LINKSTATUS_INFO_T* ptLinkStatus);

/**************************************************************************************************
 * Collision Callback
 * EPL Dropped Callback
 */

typedef void (*EPLCN_ERR_FN)(EPLCN_HANDLE_T hEplHandle, void* pvUserParam);
typedef EPLCN_ERR_FN EPLCN_COLLISION_FN;
typedef EPLCN_ERR_FN EPLCN_EPL_FRAME_DROPPED_FN;
typedef EPLCN_ERR_FN EPLCN_INTERNAL_ERROR_FN;

typedef enum EPLCN_ERROR_CALLBACK_TYPEtag
{
  EPLCN_ERR_COLLISION_CBK,
  EPLCN_ERR_EPL_FRAME_DROPPED_CBK,
  EPLCN_ERR_INTERNAL_ERROR_CBK
} EPLCN_ERROR_CALLBACK_TYPE;

void
EplCn_SetErrorCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_ERROR_CALLBACK_TYPE, EPLCN_ERR_FN, void* pvUserParam);

/**************************************************************************************************
 * MsgTypeReceived Callback
 *
 * Flags contains SoCFlags on bMsgType == SoC
 * Flags contains PReqFlags on bMsgType == PReq
 */

typedef void (* EPLCN_MSGTYPE_RECEIVED_FN)(EPLCN_HANDLE_T hEplHandle, uint8_t bMsgType, uint8_t bFlags, void* pvUserParam);

void
EplCn_SetMsgTypeRcvdCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_MSGTYPE_RECEIVED_FN, void* pvUserParam);

/**************************************************************************************************
 * LocalNodeReceived Callback
 */

typedef void (* EPLCN_MSGTYPE_RECEIVED_LOCALNODE_FN)(EPLCN_HANDLE_T hEplHandle, uint8_t bMsgType, void* pvUserParam);

void
EplCn_SetMsgTypeRcvdLocalNodeCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_MSGTYPE_RECEIVED_LOCALNODE_FN, void* pvUserParam);

/**************************************************************************************************
 * StatusFrame Buffer
 */

EPLCN_STATUS_FRAME_T* EplCn_RequestStatusFrameBufferForWriting(EPLCN_HANDLE_T hEplHandle);

EPLCN_RESULT_T
EplCn_ReleaseStatusFrameBufferForWriting(EPLCN_HANDLE_T hEplHandle, unsigned int uiNumEntries);

void
EplCn_DisableStatusFrameSending(EPLCN_HANDLE_T hEplHandle);
/* enable is done implicitly on EplCn_ReleaseStatusFrameBufferForWriting */

/**************************************************************************************************
 * IdentFrame Buffer
 */

EPLCN_IDENT_FRAME_T* EplCn_RequestIdentFrameBufferForWriting(EPLCN_HANDLE_T hEplHandle);

EPLCN_RESULT_T
EplCn_ReleaseIdentFrameBufferForWriting(EPLCN_HANDLE_T hEplHandle);

void
EplCn_DisableIdentFrameSending(EPLCN_HANDLE_T hEplHandle);
/* enable is done implicitly on EplCn_ReleaseIdentFrameBufferForWriting */

/**************************************************************************************************
 * SoC MSync Callback
 */

typedef struct EPLCN_SoC_RCVD_TIME_Ttag
{
  uint32_t ulNanosecond;
  uint32_t ulSecond;
} EPLCN_SoC_RCVD_TIME_T;

typedef void(* EPLCN_SOC_MSYNC_FN)(EPLCN_HANDLE_T hEplHandle, EPLCN_SoC_RCVD_TIME_T* ptSoCTime, void* pvUserParam);

void
EplCn_SetSoCRcvdMSyncCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_SOC_MSYNC_FN pfnCallback, void* pvUserParam);

/**************************************************************************************************
 * SoC Header Callback
 */

typedef void(* EPLCN_SOC_COM_FN)(EPLCN_HANDLE_T hEplHandle, EPLCN_SOC_HEADER_T* ptSoCTime, void* pvUserParam);

void
EplCn_SetSoCRcvdComCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_SOC_COM_FN pfnCallback, void* pvUserParam);

/**************************************************************************************************
 * SoA MSync Callback
 */

typedef void(* EPLCN_SOA_MSYNC_FN)(EPLCN_HANDLE_T hEplHandle, void* pvUserParam);

void
EplCn_SetSoARcvdMSyncCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_SOA_MSYNC_FN pfnCallback, void* pvUserParam);

/**************************************************************************************************
 * SoA Com Callback
 */

typedef void(* EPLCN_SOA_COM_FN)(EPLCN_HANDLE_T hEplHandle, EPLCN_SOA_T* ptHeader, void* pvUserParam);

void
EplCn_SetSoARcvdComCallback(EPLCN_HANDLE_T hEplHandle, EPLCN_SOA_COM_FN pfnCallback, void* pvUserParam);

/**************************************************************************************************
 * PRes interface
 */

typedef void(* EPLCN_PRES_REM_NODE_FN)(
                                      EPLCN_HANDLE_T            hEplHandle,
                                      EPLCN_POLL_RESP_FRAME_T*  ptPollRespFrame,
                                      uint32_t                  ulEthFrameSize,
                                      void*                     pvParam);

void
EplCn_SetPResRcvdRemoteNodeCallback(  EPLCN_HANDLE_T                  hEplHandle,
                                      EPLCN_PRES_REM_NODE_FN          pfnCallback,
                                      void*                           pvUserData);

#define EplCn_ReleasePResRcvdRemoteNodeFrame(hEplHandle, ptPResFrame) EplCn_ReleaseEthFrame(hEplHandle, (EPLCN_ETH_FRAME_T*) ptPResFrame)

void
EplCn_EnablePResRemoteNodeReceiving(
                          EPLCN_HANDLE_T                hEplHandle,
                          uint8_t                       bNodeId);

void
EplCn_DisablePResRemoteNodeReceiving(
                          EPLCN_HANDLE_T                hEplHandle,
                          uint8_t                       bNodeId);

/**************************************************************************************************
 * ASnd interface
 */

typedef void(* EPLCN_ASND_FN)(
                                      EPLCN_HANDLE_T            hEplHandle,
                                      EPLCN_TRANSPORT_TYPE_T    tTransportType,
                                      EPLCN_ETH_FRAME_T*        ptASndFrame,
                                      uint32_t                  ulEthFrameSize,
                                      void*                     pvParam);

void
EplCn_SetASndRcvdComCallback( EPLCN_HANDLE_T                hEplHandle,
                              EPLCN_ASND_FN                 pfnCallback,
                              void*                         pvUserData);

#define EplCn_ReleaseASndFrame(hEplHandle, ptASnd) EplCn_ReleaseEthFrame(hEplHandle, (EPLCN_ETH_FRAME_T*) ptASnd)


void
EplCn_EnableASndRemoteNodeReceiving(
                          EPLCN_HANDLE_T                hEplHandle,
                          uint8_t                       bNodeId);

void
EplCn_DisableASndRemoteNodeReceiving(
                          EPLCN_HANDLE_T                hEplHandle,
                          uint8_t                       bNodeId);

/**************************************************************************************************
 * ASnd Send interface
 */

EPLCN_ETH_FRAME_T*
EplCn_RequestASndFrame(     EPLCN_HANDLE_T  hEplHandle);

EPLCN_RESULT_T
EplCn_SendASndFrame(      EPLCN_HANDLE_T      hEplHandle,
                          EPLCN_ETH_FRAME_T*  ptASndFrame,
                          uint32_t            ulFrameLen);

/**************************************************************************************************
 * PReq Interface (own station)
 */

typedef void(* EPLCN_PREQ_MSYNC_FN)(EPLCN_HANDLE_T hEplHandle, void* pvUserParam);

void EplCn_SetPReqRcvdMSyncCallback(
                            EPLCN_HANDLE_T                hEplHandle,
                            EPLCN_PREQ_MSYNC_FN           pfnCallback,
                            void*                         pvUserParam);

/**************************************************************************************************
 * PReq Interface (own station, remote node)
 */

typedef void(* EPLCN_PREQ_COM_FN)(EPLCN_HANDLE_T hEplHandle, EPLCN_POLL_REQ_FRAME_T* ptFrame, void* pvUserParam);

void
EplCn_SetPReqRcvdComCallback( EPLCN_HANDLE_T                hEplHandle,
                              EPLCN_PREQ_COM_FN             pfnCallback,
                              void*                         pvUserParam);

#define EplCn_ReleasePReqFrame(hEplHandle, ptPReqFrame) EplCn_ReleaseEthFrame(hEplHandle, (EPLCN_ETH_FRAME_T*) ptPReqFrame)


void
EplCn_EnablePReqRemoteNodeReceiving(
                          EPLCN_HANDLE_T                hEplHandle,
                          uint8_t                       bNodeId);

void
EplCn_DisablePReqRemoteNodeReceiving(
                          EPLCN_HANDLE_T                hEplHandle,
                          uint8_t                       bNodeId);

/**************************************************************************************************
 * PRes Interface (own station)
 */

EPLCN_POLL_RESP_FRAME_T*
EplCn_RequestPResBufferForWriting(
                            EPLCN_HANDLE_T                hEplHandle);

EPLCN_RESULT
EplCn_ReleasePResBufferForWriting(EPLCN_HANDLE_T hEplHandle, uint8_t bFlags);

typedef void(* EPLCN_PRES_MSYNC_FN)(EPLCN_HANDLE_T hEplHandle, void* pvParam);

void EplCn_SetPResSentMSyncCallback(
                            EPLCN_HANDLE_T                hEplHandle,
                            EPLCN_PRES_MSYNC_FN           pfnCallback,
                            void*                         pvUserData);

void EplCn_EnablePResSending(  EPLCN_HANDLE_T                hEplHandle);

void EplCn_DisablePResSending( EPLCN_HANDLE_T                hEplHandle);

/**************************************************************************************************
 * Parameter setup
 */

void
EplCn_SetNodeAddress(   EPLCN_HANDLE_T                hEplHandle,
                        uint8_t                       bNodeAddress);

uint8_t
EplCn_GetNodeAddress(   EPLCN_HANDLE_T                hEplHandle);

/**************************************************************************************************
 * Ethernet frame interface
 */

EPLCN_ETH_FRAME_T*
EplCn_RequestEthFrame(EPLCN_HANDLE_T                hEplHandle);

void
EplCn_ReleaseEthFrame(EPLCN_HANDLE_T                hEplHandle,
                      EPLCN_ETH_FRAME_T*            ptEthFrame);

EPLCN_RESULT
EplCn_SendEthFrame(   EPLCN_HANDLE_T                hEplHandle,
                      EPLCN_ETH_FRAME_T*            ptFrame,
                      uint32_t                      ulFrameSize,
                      uint32_t                      ulIdx);

typedef void (*EPLCN_RCV_ETH_FRAME_FN)(             EPLCN_HANDLE_T                hEplHandle,
                                                    EPLCN_ETH_FRAME_T*            ptFrame,
                                                    uint32_t                      ulFrameSize,
                                                    void*                         pvUserParam);

void
EplCn_SetRecvEthFrameCallback(EPLCN_HANDLE_T                hEplHandle,
                              EPLCN_RCV_ETH_FRAME_FN        pfnEthCallback,
                              void*                         pvUserParam);

typedef void (*EPLCN_CNF_ETH_FRAME_FN)(             EPLCN_HANDLE_T                hEplHandle,
                                                    EPLCN_ETH_FRAME_T*            ptFrame,
                                                    uint32_t                      ulFrameSize,
                                                    uint32_t                      ulIdx,
                                                    void*                         pvUserParam);

void
EplCn_SetSendCnfEthFrameCallback( EPLCN_HANDLE_T                hEplHandle,
                                  EPLCN_CNF_ETH_FRAME_FN        pfnEthCallback,
                                  void*                         pvUserParam);

/******************************************************************************
 * Error Signalling flags
 */
int
EplCn_GetExceptionAckBit(   EPLCN_HANDLE_T                      hEplHandle);

int
EplCn_GetExceptionResetBit( EPLCN_HANDLE_T                      hEplHandle);

void
EplCn_SetExceptionNewBit(   EPLCN_HANDLE_T                      hEplHandle,
                            int                                 fBitValue);

void
EplCn_SetExceptionClearBit( EPLCN_HANDLE_T                      hEplHandle,
                            int                                 fBitValue);

#endif
